function [handle_oscilloscope_figure, handle_plot_lines] = create_figure_with_controls()
% CREATE_FIGURE_WITH_CONTROLS - a function that creates a GUI for the
% oscilloscope
%
% Syntax: [handle_oscilloscope_figure, handle_plot_lines] = create_figure_with_controls()
%
% Inputs:
%    function does not take any input
%
% Outputs:
%    handle_oscilloscope_figure    - handle of the GUI figure
%    handle_plot_lines             - handle of the graph lines
%
% Other m-files required: change_sound_card
% Subfunctions: none
% Callback functions: callback_check_channel, callback_freq_change,
%                     callback_amplitude_change, callback_trigger_change,
%                     callback_trigger_channel_change, callback_pause_button
% MAT-files required: none
%
% Author: Antonin Novak
% Laboratoire d'Acoustique de l'Université du Mans
% (LAUM, UMR CNRS 6613), 72085 Le Mans, France.
% email address: antonin.novak@univ-lemans.fr  
% Website: https://ant-novak.com
% October 2019; Last revision: 09-Jan-2020

%% ------------- BEGIN CODE --------------

global IMDEA_sound_card IMDEA_generator

% create a new figure // when pressed close - stop PLAY and delete figure
handle_oscilloscope_figure  = figure('CloseRequestFcn', 'PLAY = 0;', 'name', 'Oscilloscope v1.0','NumberTitle','off');

% create a new axes (inside the figure and leave space for uicontrols, ...)
axes ('OuterPosition', [0 0.4 1 0.6]);


%% checkbox for each channel

for n = 1:length(IMDEA_sound_card.in_channels)
    
    uicontrol(handle_oscilloscope_figure, ...
        'Style'         , 'checkbox', ...
        'String'        , ['ch. ' num2str(IMDEA_sound_card.in_channels(n))], ...
        'CallBack'      , {@callback_check_channel, n}, ...
        'Value'         , 1, ...
        'Units'         , 'normalized', ...
        'Position'      , [0.91 0.95-n*0.05 0.09 0.05]);
    
end

%% panels
pannel_1 = uipanel(handle_oscilloscope_figure, ...
    'Title'              , 'Signal Generator', ...
    'Units'              , 'normalized', ...
    'Position'           , [0.09 0.1 0.4 0.3]);

pannel_2 = uipanel(handle_oscilloscope_figure, ...
    'Title'              , 'Oscilloscope', ...
    'Units'              , 'normalized', ...
    'Position'           , [0.51 0.1 0.4 0.3]);




%% FREQUENCY settings
uicontrol(pannel_1, ...
    'Style'              , 'text', ...
    'String'             , 'Frequency :', ...
    'Units'              , 'normalized', ...
    'Position'           , [0.1 0.7 0.3 0.2], ...
    'HorizontalAlignment', 'right');

uicontrol(pannel_1, ...
    'Style'              , 'edit', ...
    'CallBack'           , @callback_freq_change, ...
    'Tag'                , 'freq_edit', ...
    'String'             , num2str(IMDEA_generator.f), ...
    'Units'              , 'normalized', ...
    'Position'           , [0.41 0.71 0.3 0.2], ...
    'HorizontalAlignment', 'left');

uicontrol(pannel_1, ...
    'Style'              , 'text', ...
    'String'             , 'Hz', ...
    'Units'              , 'normalized', ...
    'Position'           , [0.72 0.7 0.2 0.2], ...
    'HorizontalAlignment', 'left');


%% AMPLITUDE settings
uicontrol(pannel_1, ...
    'Style'              , 'text', ...
    'String'             , 'Amplitude :', ...
    'Units'              , 'normalized', ...
    'Position'           , [0.1 0.5 0.3 0.2], ...
    'HorizontalAlignment', 'right');

uicontrol(pannel_1, ...
    'Style'              , 'edit', ...
    'CallBack'           , @callback_amplitude_change, ...
    'Tag'                , 'amplitude_edit', ...
    'String'             , num2str(IMDEA_generator.A), ...
    'Units'              , 'normalized', ...
    'Position'           , [0.41 0.51 0.3 0.2], ...
    'HorizontalAlignment', 'left');



%% TRIGGER
uicontrol(pannel_2, ...
    'Style'              , 'Popup', ...
    'CallBack'           , @callback_trigger_change, ...
    'String'             , {'Trig. off', 'Trig. generator', 'Trig. ext'}, ...
    'Units'              , 'normalized', ...
    'Position'           , [0.05 0.7 0.9 0.2]);


% create pop-up menu of channels (cell of strings)
channel_strings = cell(1,length(IMDEA_sound_card.in_channels));
for n = 1:length(IMDEA_sound_card.in_channels)
    channel_strings{n} = ['ch. ' num2str(IMDEA_sound_card.in_channels(n))];
    %'ch. 1'
    %'ch. 2'
end

uicontrol(pannel_2, ...
    'Style'              , 'Popup', ...
    'CallBack'           , @callback_trigger_channel_change, ...
    'String'             , channel_strings, ...
    'Tag'                , 'trigger_channel', ...
    'Enable'             , 'off', ...
    'Units'              , 'normalized', ...
    'Position'           , [0.05 0.4 0.9 0.2]);

uicontrol(pannel_2, ...
    'Style'              , 'pushbutton', ...
    'CallBack'           , @callback_pause_button, ...
    'String'             , 'Pause', ...
    'BackgroundColor'    , [0 1 0], ...
    'Units'              , 'normalized', ...
    'Position'           , [0.05 0.1 0.9 0.2]);


%%


% create new plot lines (for each channel)
zeros_vector = zeros(IMDEA_sound_card.BufferSize, length(IMDEA_sound_card.in_channels));
handle_plot_lines = plot(zeros_vector, 'linewidth', 2);

% set axes properties
xlim([0 IMDEA_sound_card.BufferSize]);
grid on
legend(channel_strings);


%% ------------- END CODE --------------


