function change_sound_card()
% CHANGE_SOUND_CARD - a function that saves the name of selected sound
% card to a data file 
% Syntax:  change_sound_card()
%
% Inputs:
%    function does not take any input
%
% Outputs:
%    function does not provide any output
%
% Other m-files required: get_list_of_soundcards.m
% Subfunctions: selection_player, selection_recorder
% MAT-files required: 'data/last_used_soundcard.mat'
%
% Author: Antonin Novak
% Laboratoire d'Acoustique de l'Université du Mans
% (LAUM, UMR CNRS 6613), 72085 Le Mans, France.
% email address: antonin.novak@univ-lemans.fr  
% Website: https://ant-novak.com
% October 2019; Last revision: 27-Oct-2019

%% ------------- BEGIN CODE --------------

% file name of the data file
data_file_name = 'data/last_used_soundcard.mat';

% load names of last used soundcard_player and soundcard_recorder
load('data/last_used_soundcard.mat', 'DeviceName_player', 'DeviceName_recoreder');

% create a new figure
handle_sundcard_figure = figure(...
    'Units'        , 'normalized', ...
    'Position'     , [0.4 0.3 0.3 0.2], ...
    'Name'         , 'Sound Card Selector', ...
    'NumberTitle'  , 'off', ...
    'ToolBar'      , 'none', ...
    'MenuBar'      , 'none', ...
    'WindowStyle'  , 'modal');

% get list of available sound cards (input and output)
[device_names_players, device_names_recorders] = get_list_of_soundcards();

% find the last used sound card output and set the index of the list
[idx_last_sound_player,~]   = find(ismember(device_names_players, DeviceName_player));
if isempty(idx_last_sound_player)
    idx_last_sound_player = 1;
end

% find the last used sound card input and set the index of the list
[idx_last_sound_recorder,~] = find(ismember(device_names_recorders, DeviceName_recoreder));
if isempty(idx_last_sound_recorder)
    idx_last_sound_recorder = 1;
end

%% Save the sound cards (before selecting)
% this part of code is run in case a user closes the window without
% selecting the sound card.
DeviceName_player    = device_names_players{idx_last_sound_player};
DeviceName_recoreder = device_names_recorders{idx_last_sound_recorder};
save(data_file_name, 'DeviceName_player',    '-append');
save(data_file_name, 'DeviceName_recoreder', '-append');



%% Sound Player selector
% text 'Sound Player'
uicontrol(handle_sundcard_figure, ...
    'Style'     , 'text', ...
    'Units'     , 'normalized', ...
    'Position'  , [0.1 0.7 0.2, 0.1], ...
    'String'    , 'Sound Player');

% popup menu of available output sound cards
uicontrol(handle_sundcard_figure, ...
    'Style'     , 'popupmenu', ...
    'Units'     , 'normalized', ...
    'Position'  , [0.3 0.7 0.5, 0.1], ...
    'String'    , device_names_players, ...
    'Value'     , idx_last_sound_player, ...
    'Callback'  , @selection_player);

    % function callback when selecting an output sound card
    % -- save the name of the output sound card
    function selection_player(src,~)
        try
            DeviceName_player = src.String{src.Value};
            save(data_file_name, 'DeviceName_player', '-append');
            disp([DeviceName_player ' was saved as a default sound recorder']);
        catch
            err('ERROR: Not able to save the selected card to data/last_used_soundcard.mat');
        end
    end


%% Sound Recorder selector
% text 'Sound Recorder'
uicontrol(handle_sundcard_figure, ...
    'Style'     , 'text', ...
    'Units'     , 'normalized', ...
    'Position'  , [0.1 0.4 0.2, 0.1], ...
    'String'    , 'Sound Recorder');

% popup menu of available input sound cards
uicontrol(handle_sundcard_figure, ...
    'Style'     , 'popupmenu', ...
    'Units'     , 'normalized', ...
    'Position'  , [0.3 0.4 0.5, 0.1], ...
    'String'    , device_names_recorders, ...
    'Value'     , idx_last_sound_recorder, ...
    'Callback'  , @selection_recorder);

    % function callback when selecting an input sound card
    % -- save the name of the input sound card
    function selection_recorder(src,~)
        try
            DeviceName_recoreder = src.String{src.Value};
            save(data_file_name, 'DeviceName_recoreder', '-append');
            disp([DeviceName_recoreder ' was saved as a default sound recorder']);
        catch
            err('ERROR: Not able to save the selected card to data/last_used_soundcard.mat');
        end
    end


%% OK button
uicontrol(handle_sundcard_figure, ...
    'Style'     , 'pushbutton', ...
    'Units'     , 'normalized', ...
    'Position'  , [0.3 0.1 0.3, 0.15], ...
    'String'    , 'OK', ...
    'Callback'  , @ok_button);

    % function callback: close the window
    function ok_button(~,~)
        delete(handle_sundcard_figure);
    end

%% wait until the figrue closes
uiwait(handle_sundcard_figure);

end

%% ------------- END CODE --------------

